<?php
// File: /public_html/sm.okgroups.com/app/Http/Controllers/Api/App/AppContentsController.php

namespace App\Http\Controllers\Api\App;

use App\Http\Controllers\Controller;
use App\Models\AppContent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;

class AppContentsController extends Controller
{
    /**
     * Determine requested language and build a fallback chain:
     * e.g. "pt-BR" -> ["pt-br", "pt", "en"]
     *
     * ✅ UPDATED: NO FALLBACK
     * We return ONLY the requested lang (normalized), and if missing -> []
     */
    private function langChain(Request $request): array
    {
        // $defaultLang = strtolower((string) config('app.fallback_locale', 'en')); // ❌ removed fallback
        $raw = (string) $request->query('lang', '');

        // ✅ If no explicit lang -> return EMPTY chain (no fallback)
        if (trim($raw) === '') {
            return [];
        }

        $lang = strtolower(str_replace('_', '-', trim($raw)));

        $chain = [];
        if ($lang !== '') {
            $chain[] = $lang;

            // ❌ removed base language fallback (pt-br -> pt)
            // if (str_contains($lang, '-')) {
            //     $base = explode('-', $lang, 2)[0];
            //     if ($base !== '' && $base !== $lang) {
            //         $chain[] = $base;
            //     }
            // }
        }

        // ❌ removed default language fallback (en)
        // if (!in_array($defaultLang, $chain, true)) {
        //     $chain[] = $defaultLang;
        // }

        // De-dup
        $chain = array_values(array_unique($chain));

        return $chain;
    }

    public function index(Request $request)
    {
        $hasLang = Schema::hasColumn('app_contents', 'language_code');
        $chain = $this->langChain($request);

        // ✅ Old behavior: when no ?lang= , return only default language (like before)
        // ✅ UPDATED: if no ?lang= -> return EMPTY (no fallback)
        $langWasProvided = trim((string) $request->query('lang', '')) !== '';

        if (!$langWasProvided) {
            return response()->json(['items' => (object)[]]);
        }

        if (!$hasLang) {
            // No language support in DB yet => keep old behavior exactly
            $rows = AppContent::query()
                ->select(['key','title','body','updated_at'])
                ->get();

            $items = [];
            foreach ($rows as $r) {
                $items[$r->key] = [
                    'title' => $r->title,
                    'body' => $r->body,
                    'updated_at' => optional($r->updated_at)->toIso8601String(),
                ];
            }

            return response()->json(['items' => $items]);
        }

        // ✅ UPDATED: if chain is empty (shouldn't happen when lang provided), return empty
        if (count($chain) === 0) {
            return response()->json(['items' => (object)[]]);
        }

        // lang requested => load ONLY requested language (no fallback)
        $rows = AppContent::query()
            ->whereIn('language_code', $chain)
            ->select(['key','title','body','updated_at','language_code'])
            ->get();

        $items = [];
        foreach ($rows as $r) {
            $items[$r->key] = [
                'title' => $r->title,
                'body' => $r->body,
                'updated_at' => optional($r->updated_at)->toIso8601String(),
            ];
        }

        return response()->json(['items' => (object)$items]);
    }

    public function show(Request $request, string $key)
    {
        $hasLang = Schema::hasColumn('app_contents', 'language_code');
        $chain = $this->langChain($request);
        $langWasProvided = trim((string) $request->query('lang', '')) !== '';

        if (!$hasLang) {
            // No language support in DB yet => keep old behavior exactly
            $r = AppContent::query()
                ->where('key', $key)
                ->select(['key','title','body','updated_at'])
                ->firstOrFail();

            return response()->json([
                'key' => $r->key,
                'title' => $r->title,
                'body' => $r->body,
                'updated_at' => optional($r->updated_at)->toIso8601String(),
            ]);
        }

        // ✅ UPDATED: if no lang provided -> return EMPTY (no fallback)
        if (!$langWasProvided) {
            return response()->json([
                'key' => $key,
                'title' => '',
                'body' => '',
                'updated_at' => null,
            ]);
        }

        // ✅ UPDATED: if chain empty -> return EMPTY
        if (count($chain) === 0) {
            return response()->json([
                'key' => $key,
                'title' => '',
                'body' => '',
                'updated_at' => null,
            ]);
        }

        // lang requested => ONLY exact lang (no base, no default)
        $r = AppContent::query()
            ->where('key', $key)
            ->whereIn('language_code', $chain)
            ->select(['key','title','body','updated_at','language_code'])
            ->first();

        if (!$r) {
            // ✅ UPDATED: return EMPTY instead of fallback
            return response()->json([
                'key' => $key,
                'title' => '',
                'body' => '',
                'updated_at' => null,
            ]);
        }

        return response()->json([
            'key' => $r->key,
            'title' => $r->title,
            'body' => $r->body,
            'updated_at' => optional($r->updated_at)->toIso8601String(),
        ]);
    }
}