<?php
// File: /home/okgroups/public_html/sm.okgroups.com/app/Http/Controllers/Api/App/VoucherRedeemController.php

namespace App\Http\Controllers\Api\App;

use App\Http\Controllers\Controller;
use App\Models\AppContent;
use App\Models\Voucher;
use App\Models\VoucherRedemption;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class VoucherRedeemController extends Controller
{
    public function redeem(Request $request)
    {
        $validated = $request->validate([
            'code' => ['required', 'string', 'max:64'],
            'device_id' => ['nullable', 'string', 'max:128'],
            'user_id' => ['nullable', 'integer'],
        ]);

        $code = strtoupper(trim($validated['code']));
        $deviceId = $validated['device_id'] ?? null;

        // اگر کاربر لاگین باشد، همان را منبع حقیقت قرار بده
        $authUser = $request->user();
        $userId = $authUser?->id ?? ($validated['user_id'] ?? null);

        // ✅ SAFE: do not assume column name in app_contents
        $row = AppContent::query()->where('key', 'voucher_popup')->first();

        $popupTemplate =
            ($row->content ?? null)
            ?? ($row->value ?? null)
            ?? ($row->body ?? null)
            ?? ($row->text ?? null)
            ?? 'Voucher {code} is invalid or already used.';

        $ip = $request->ip();
        $ua = substr((string) $request->userAgent(), 0, 255);

        // پیام‌های دقیق‌تر (فارسی)
        $reasonMsg = function (string $reason) use ($code, $popupTemplate) {
            $fallback = str_replace('{code}', $code, (string) $popupTemplate);

            return match ($reason) {
                'invalid'  => 'این کد در سیستم وجود ندارد.',
                'inactive' => 'این کد غیرفعال است.',
                'expired'  => 'اعتبار این کد به پایان رسیده است.',
                'used'     => 'این کد قبلاً استفاده شده است.',
                'need_login' => 'برای استفاده از کد، ابتدا وارد حساب شوید.',
                'credit_schema' => 'خطای سیستم شارژ. لطفاً با پشتیبانی تماس بگیرید.',
                default    => $fallback,
            };
        };

        $fail = function (?Voucher $voucher, string $reason) use ($code, $deviceId, $userId, $ip, $ua, $reasonMsg) {
            VoucherRedemption::create([
                'voucher_id' => $voucher?->id,
                'code_snapshot' => $code,
                'user_id' => $userId,
                'device_id' => $deviceId,
                'status' => 'failed',
                'fail_reason' => $reason,
                'redeemed_at' => now(),
                'ip' => $ip,
                'user_agent' => $ua,
            ]);

            return response()->json([
                'ok' => false,
                'message' => $reasonMsg($reason),
                'reason' => $reason,
            ]);
        };

        // قبل از هرچیز: اگر user نداریم، اجازه نده که voucher used شود
        if (!$userId) {
            return $fail(null, 'need_login');
        }

        $voucher = Voucher::query()->where('code', $code)->first();

        if (!$voucher) {
            return $fail(null, 'invalid');
        }

        if (!$voucher->is_active) {
            return $fail($voucher, 'inactive');
        }

        if ($voucher->isExpired()) {
            return $fail($voucher, 'expired');
        }

        return DB::transaction(function () use ($voucher, $code, $deviceId, $userId, $ip, $ua, $fail) {

            // قفل روی voucher
            $locked = Voucher::query()->where('id', $voucher->id)->lockForUpdate()->first();

            if (!$locked) {
                return $fail(null, 'invalid');
            }

            if (!$locked->is_active) {
                return $fail($locked, 'inactive');
            }

            if ($locked->isExpired()) {
                return $fail($locked, 'expired');
            }

            if ($locked->isUsedUp()) {
                return $fail($locked, 'used');
            }

            $secondsToAdd = (int) ($locked->seconds ?? 0);
            if ($secondsToAdd <= 0) {
                // اگر voucher تایم ندارد، اصلاً موفق نکن
                return $fail($locked, 'credit_schema');
            }

            // ✅ امن: نام ستون credit را پیدا کن
            $creditCol = null;
            if (Schema::hasColumn('users', 'credit_seconds')) {
                $creditCol = 'credit_seconds';
            } elseif (Schema::hasColumn('users', 'remaining_seconds')) {
                $creditCol = 'remaining_seconds';
            }

            if (!$creditCol) {
                // اگر ساختار credit در users مشخص نیست، موفق نکن و voucher هم used نشود
                return $fail($locked, 'credit_schema');
            }

            // قفل روی user و افزایش credit (اتمیک)
            DB::table('users')
                ->where('id', $userId)
                ->lockForUpdate()
                ->update([
                    $creditCol => DB::raw("$creditCol + {$secondsToAdd}"),
                    'credit_updated_at' => now(),
                    'updated_at' => now(),
                ]);

            // فقط بعد از موفقیت شارژ، voucher را used کن
            $locked->increment('uses_count');

            VoucherRedemption::create([
                'voucher_id' => $locked->id,
                'code_snapshot' => $code,
                'user_id' => $userId,
                'device_id' => $deviceId,
                'status' => 'success',
                'fail_reason' => null,
                'redeemed_at' => now(),
                'ip' => $ip,
                'user_agent' => $ua,
            ]);

            // مقدار جدید credit را برگردان (برای UI)
            $newCredit = (int) DB::table('users')->where('id', $userId)->value($creditCol);

            // hms ساده
            $h = intdiv($newCredit, 3600);
            $m = intdiv($newCredit % 3600, 60);
            $s = $newCredit % 60;
            $hms = sprintf('%02d:%02d:%02d', $h, $m, $s);

            return response()->json([
                'ok' => true,
                'message' => 'کد با موفقیت اعمال شد.',
                'seconds_added' => $secondsToAdd,
                'credit_seconds' => $newCredit,
                'credit_hms' => $hms,
                'code' => $code,
            ]);
        });
    }
}
