<?php
// File: /home/okgroups/public_html/sm.okgroups.com/app/Http/Controllers/Api/Auth/DeviceAuthController.php

namespace App\Http\Controllers\Api\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Schema;

class DeviceAuthController extends Controller
{
    public function login(Request $request)
    {
        $data = $request->validate([
            'device_id'   => ['required', 'string', 'min:6', 'max:64', 'regex:/^[a-zA-Z0-9_-]+$/'],
            'device_name' => ['nullable', 'string', 'max:64'],
        ]);

        $deviceId   = trim($data['device_id']);
        $deviceName = $data['device_name'] ?? 'SM-Android';

        // username سیستمی بر اساس device_id
        $username = 'd_' . $deviceId;

        // پسورد داخلی (ثابت) - فقط برای ذخیره در DB (برای سازگاری)
        $secret  = config('app.key') ?: env('APP_KEY', 'fallback');
        $rawPass = hash('sha256', $deviceId . '|' . $secret);
        $rawPass = substr($rawPass, 0, 32);

        // پیدا کردن یوزر بر اساس device_id (اگر ستونش هست) وگرنه بر اساس username
        $q = User::query();
        if (Schema::hasColumn('users', 'device_id')) {
            $q->where('device_id', $deviceId);
        } else {
            $q->where('username', $username);
        }

        $user = $q->first();

        if (!$user) {
            $user = new User();

            if (Schema::hasColumn('users', 'device_id')) {
                $user->device_id = $deviceId;
            }

            $user->username = $username;
            $user->email    = $username . '@device.local';

            if (Schema::hasColumn('users', 'first_name')) $user->first_name = 'Device';
            if (Schema::hasColumn('users', 'last_name'))  $user->last_name  = $deviceId;

            $user->password = Hash::make($rawPass);

            if (Schema::hasColumn('users', 'email_verified_at')) $user->email_verified_at = now();
            if (Schema::hasColumn('users', 'status')) $user->status = 'Active';
            if (Schema::hasColumn('users', 'active')) $user->active = 1;

            // role_id اجباری در Vanguard
            if (Schema::hasColumn('users', 'role_id') && empty($user->role_id)) {
                $firstRoleId = DB::table('roles')->orderBy('id')->value('id');
                if ($firstRoleId) $user->role_id = $firstRoleId;
            }

            $user->save();
        } else {
            // همس��ن‌سازی پسورد تا همیشه ثابت باشد
            if (!Hash::check($rawPass, $user->password)) {
                $user->password = Hash::make($rawPass);
            }

            if (Schema::hasColumn('users', 'device_id') && empty($user->device_id)) {
                $user->device_id = $deviceId;
            }

            if (Schema::hasColumn('users', 'role_id') && empty($user->role_id)) {
                $firstRoleId = DB::table('roles')->orderBy('id')->value('id');
                if ($firstRoleId) $user->role_id = $firstRoleId;
            }

            $user->save();
        }

        // ✅ ساخت/تثبیت public_code (۷ رقمی) برای نمایش به کاربر
        if (Schema::hasColumn('users', 'public_code') && empty($user->public_code)) {
            $user->public_code = $this->generateUniquePublicCode();
            $user->save();
        }

        // ✅ اطمینان از اینکه credit_seconds / credit_updated_at مقدار دارند
        if (Schema::hasColumn('users', 'credit_seconds')) {
            $changed = false;

            if ($user->credit_seconds === null) {
                // ✅ دریافت دقیقه پاداش از settings
                $bonusMinutes = (int) app_setting('signup_bonus_minutes', 3);
                $bonusSeconds = $bonusMinutes * 60;
                
                $user->credit_seconds = $bonusSeconds;
                $changed = true;
            }

            if (Schema::hasColumn('users', 'credit_updated_at') && empty($user->credit_updated_at)) {
                $user->credit_updated_at = now();
                $changed = true;
            }

            if ($changed) {
                $user->save();
            }
        }

        // ✅ توکن مستقیم با Sanctum (بدون AuthController/Vanguard)
        $token = $user->createToken($deviceName)->plainTextToken;

        $creditSeconds = Schema::hasColumn('users', 'credit_seconds') ? (int)($user->credit_seconds ?? 0) : null;
        $creditHms = ($creditSeconds !== null) ? $this->secondsToHms($creditSeconds) : null;

        $userPayload = [
            'id' => $user->id,
            'username' => $user->username,
            'public_code' => $user->public_code ?? null,
        ];

        if ($creditSeconds !== null) {
            $userPayload['credit_seconds'] = $creditSeconds;
            $userPayload['credit_hms'] = $creditHms;
            if (Schema::hasColumn('users', 'credit_updated_at')) {
                $userPayload['credit_updated_at'] = optional($user->credit_updated_at)->toISOString();
            }
        }

        return response()->json([
            'token' => $token,
            'token_type' => 'Bearer',
            'user' => $userPayload,
        ], 200);
    }

    private function generateUniquePublicCode(): int
    {
        // 7-digit: 1000000..9999999
        do {
            $code = random_int(1000000, 9999999);
            $exists = DB::table('users')->where('public_code', $code)->exists();
        } while ($exists);

        return $code;
    }

    private function secondsToHms(int $seconds): string
    {
        if ($seconds < 0) $seconds = 0;
        $h = intdiv($seconds, 3600);
        $m = intdiv($seconds % 3600, 60);
        $s = $seconds % 60;
        return sprintf('%02d:%02d:%02d', $h, $m, $s);
    }
}