<?php
// File: /home/okgroups/public_html/sm.okgroups.com/app/Http/Controllers/Api/Profile/CreditController.php

namespace App\Http\Controllers\Api\Profile;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;
use Carbon\Carbon; // ✅ NEW

class CreditController extends Controller
{
    /**
     * GET /api/me/credit
     * برمی‌گرداند: credit_seconds + credit_hms + credit_updated_at (ISO8601)
     */
    public function show(Request $request)
    {
        $user = $request->user();

        $sec = 0;

        if (Schema::hasColumn('users', 'credit_seconds')) {
            $sec = (int) ($user->credit_seconds ?? 0);
        }

        return response()->json([
            'credit_seconds'    => $sec,
            'credit_hms'        => $this->secToHms($sec),
            'credit_updated_at' => $this->iso($user),
        ], 200);
    }

    /**
     * PUT /api/me/credit
     * ورودی: credit_seconds یا credit_hms (مثل 01:27:11)
     * خروجی: مقدار ذخیره‌شده
     */
    public function update(Request $request)
    {
        $user = $request->user();

        $data = $request->validate([
            'credit_seconds' => ['nullable', 'integer', 'min:0', 'max:315360000'], // تا 10 سال
            'credit_hms'     => ['nullable', 'string', 'regex:/^\d{2}:\d{2}:\d{2}$/'],
        ]);

        $sec = null;

        if (array_key_exists('credit_seconds', $data) && $data['credit_seconds'] !== null) {
            $sec = (int) $data['credit_seconds'];
        } elseif (!empty($data['credit_hms'])) {
            $sec = $this->hmsToSec($data['credit_hms']);
        } else {
            return response()->json([
                'message' => 'Either credit_seconds or credit_hms is required.',
                'errors'  => [
                    'credit_seconds' => ['Provide credit_seconds or credit_hms.'],
                ],
            ], 422);
        }

        if (!Schema::hasColumn('users', 'credit_seconds')) {
            return response()->json([
                'message' => 'credit_seconds column is missing in users table.',
            ], 500);
        }

        $user->credit_seconds = $sec;

        if (Schema::hasColumn('users', 'credit_updated_at')) {
            $user->credit_updated_at = now();
        }

        $user->save();

        return response()->json([
            'credit_seconds'    => (int) $user->credit_seconds,
            'credit_hms'        => $this->secToHms((int) $user->credit_seconds),
            'credit_updated_at' => $this->iso($user),
        ], 200);
    }

    /**
     * POST /api/me/credit/add
     * ورودی: seconds (int) یا hms (HH:MM:SS)
     * کار: به credit_seconds اضافه می‌کند
     */
    public function add(Request $request)
    {
        $user = $request->user();

        $data = $request->validate([
            'seconds' => ['nullable', 'integer', 'min:0', 'max:315360000'],
            'hms'     => ['nullable', 'string', 'regex:/^\d{2}:\d{2}:\d{2}$/'],
        ]);

        $add = null;

        if (array_key_exists('seconds', $data) && $data['seconds'] !== null) {
            $add = (int) $data['seconds'];
        } elseif (!empty($data['hms'])) {
            $add = $this->hmsToSec($data['hms']);
        } else {
            return response()->json([
                'message' => 'Either seconds or hms is required.',
                'errors'  => [
                    'seconds' => ['Provide seconds or hms.'],
                ],
            ], 422);
        }

        if (!Schema::hasColumn('users', 'credit_seconds')) {
            return response()->json([
                'message' => 'credit_seconds column is missing in users table.',
            ], 500);
        }

        $current = (int) ($user->credit_seconds ?? 0);
        $new = $current + $add;

        $user->credit_seconds = $new;

        if (Schema::hasColumn('users', 'credit_updated_at')) {
            $user->credit_updated_at = now();
        }

        $user->save();

        return response()->json([
            'credit_seconds'    => (int) $user->credit_seconds,
            'credit_hms'        => $this->secToHms((int) $user->credit_seconds),
            'credit_updated_at' => $this->iso($user),
        ], 200);
    }

    /**
     * POST /api/me/credit/consume
     * ورودی: seconds (int) یا hms (HH:MM:SS)
     * کار: از credit_seconds کم می‌کند (کفِ صفر)
     */
    public function consume(Request $request)
    {
        $user = $request->user();

        $data = $request->validate([
            'seconds' => ['nullable', 'integer', 'min:0', 'max:315360000'],
            'hms'     => ['nullable', 'string', 'regex:/^\d{2}:\d{2}:\d{2}$/'],
        ]);

        $consume = null;

        if (array_key_exists('seconds', $data) && $data['seconds'] !== null) {
            $consume = (int) $data['seconds'];
        } elseif (!empty($data['hms'])) {
            $consume = $this->hmsToSec($data['hms']);
        } else {
            return response()->json([
                'message' => 'Either seconds or hms is required.',
                'errors'  => [
                    'seconds' => ['Provide seconds or hms.'],
                ],
            ], 422);
        }

        if (!Schema::hasColumn('users', 'credit_seconds')) {
            return response()->json([
                'message' => 'credit_seconds column is missing in users table.',
            ], 500);
        }

        $current = (int) ($user->credit_seconds ?? 0);
        $new = max(0, $current - $consume);

        $user->credit_seconds = $new;

        if (Schema::hasColumn('users', 'credit_updated_at')) {
            $user->credit_updated_at = now();
        }

        $user->save();

        return response()->json([
            'credit_seconds'    => (int) $user->credit_seconds,
            'credit_hms'        => $this->secToHms((int) $user->credit_seconds),
            'credit_updated_at' => $this->iso($user),
        ], 200);
    }

    private function iso($user): ?string
    {
        if (!Schema::hasColumn('users', 'credit_updated_at')) {
            return null;
        }

        $v = $user->credit_updated_at ?? null;
        if (!$v) return null;

        // اگر Carbon/DateTime هست
        if (is_object($v) && method_exists($v, 'toISOString')) {
            return $v->utc()->toISOString();
        }

        // اگر string هست => به ISO8601 تبدیلش کن
        try {
            return Carbon::parse((string) $v)->utc()->toISOString();
        } catch (\Throwable $e) {
            // fail-safe
            return (string) $v;
        }
    }

    private function secToHms(int $sec): string
    {
        $sec = max(0, $sec);
        $h = intdiv($sec, 3600);
        $m = intdiv($sec % 3600, 60);
        $s = $sec % 60;
        return sprintf('%02d:%02d:%02d', $h, $m, $s);
    }

    private function hmsToSec(string $hms): int
    {
        [$h, $m, $s] = array_map('intval', explode(':', $hms));
        $h = max(0, $h);
        $m = min(max(0, $m), 59);
        $s = min(max(0, $s), 59);
        return ($h * 3600) + ($m * 60) + $s;
    }
}
