<?php

namespace App\Http\Controllers;

use App\Models\FraudAlert;
use Illuminate\Http\Request;

class FraudAlertController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $alerts = FraudAlert::with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(15);
        
        return view('fraud-alerts.index', compact('alerts'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('fraud-alerts.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|integer|exists:users,id',
            'device_id' => 'nullable|string|max:255',
            'user_ip' => 'nullable|ip',
            'alert_type' => 'required|string|max:100',
            'description' => 'nullable|string',
            'severity' => 'required|in:low,medium,high',
            'status' => 'required|in:pending,investigating,resolved'
        ]);

        FraudAlert::create($validated);
        
        return redirect()->route('fraud-alerts.index')
            ->with('success', 'Fraud alert created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(FraudAlert $fraudAlert)
    {
        $fraudAlert->load('user');
        return view('fraud-alerts.show', compact('fraudAlert'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(FraudAlert $fraudAlert)
    {
        return view('fraud-alerts.edit', compact('fraudAlert'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, FraudAlert $fraudAlert)
    {
        $validated = $request->validate([
            'user_id' => 'required|integer|exists:users,id',
            'device_id' => 'nullable|string|max:255',
            'user_ip' => 'nullable|ip',
            'alert_type' => 'required|string|max:100',
            'description' => 'nullable|string',
            'severity' => 'required|in:low,medium,high',
            'status' => 'required|in:pending,investigating,resolved',
            'resolved_at' => 'nullable|date'
        ]);

        if ($validated['status'] === 'resolved' && !$validated['resolved_at']) {
            $validated['resolved_at'] = now();
        }

        $fraudAlert->update($validated);
        
        return redirect()->route('fraud-alerts.show', $fraudAlert->id)
            ->with('success', 'Fraud alert updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(FraudAlert $fraudAlert)
    {
        $fraudAlert->delete();
        
        return redirect()->route('fraud-alerts.index')
            ->with('success', 'Fraud alert deleted successfully.');
    }
}