<?php

namespace App\Http\Controllers;

use App\Models\Purchase;
use App\Models\User;
use App\Models\FraudAlert;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Google\Client as GoogleClient;
use Google\Service\AndroidPublisher;

class PurchaseController extends Controller
{
    /**
     * ثبت و تایید خرید با اعتبارسنجی گوگل پلی
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'user_id' => 'nullable|exists:users,public_code',
            'device_id' => 'nullable|string|max:128',
            'product_id' => 'required|string|max:128',
            'receipt' => 'required|string',            // purchaseToken
            'purchase_token' => 'nullable|string|max:512',
            'seconds_added' => 'required|integer|min:0',
            'price' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|size:3',
        ]);

        $user = null;
        if ($validated['user_id']) {
            $user = User::where('public_code', $validated['user_id'])->first();
        }

        // ---------- اعتبارسنجی خرید با گوگل پلی ------------
        $packageName   = 'us.smartmeet.app'; // ← مقدار متناسب با اپ خودت!
        $productId     = $validated['product_id'];
        $purchaseToken = $validated['receipt'];

        $googleVerify = $this->verifyPurchaseWithGooglePlay(
            $packageName,
            $productId,
            $purchaseToken
        );

        if (!$googleVerify['success']) {
            return response()->json([
                'error' => 'خرید توسط گوگل پلی تایید نشد!',
                'google_response' => $googleVerify['error'] ?? null,
            ], 400);
        }
        // ---------------------------------------------------

        $purchase = Purchase::create([
            'user_id'        => $user?->id,
            'device_id'      => $validated['device_id'] ?? null,
            'product_id'     => $validated['product_id'],
            'receipt'        => $validated['receipt'],
            'purchase_token' => $validated['purchase_token'] ?? null,
            'seconds_added'  => $validated['seconds_added'],
            'status'         => 'success',
            'price'          => $validated['price'] ?? null,
            'currency'       => $validated['currency'] ?? 'USD',
            'ip'             => $request->ip(),
            'user_agent'     => $request->userAgent(),
            'purchased_at'   => now(),
        ]);

        if ($user) {
            $user->increment('credit_seconds', $validated['seconds_added']);
            $user->update(['credit_updated_at' => now()]);
        }

        return response()->json([
            'message'  => 'Purchase recorded successfully',
            'purchase' => $purchase,
        ], 201);
    }

    /**
     * اعتبارسنجی خرید با Google Play Developer API
     */
    private function verifyPurchaseWithGooglePlay(string $packageName, string $productId, string $purchaseToken): array
    {
        try {
            $client = new GoogleClient();
            // مسیر فایل Service Account گوگل. حتماً فایل و مسیر را درست قرار بده!
            $client->setAuthConfig(storage_path('app/play-verifier.json'));
            $client->addScope(AndroidPublisher::ANDROIDPUBLISHER);

            $service = new AndroidPublisher($client);

            $purchase = $service->purchases_products->get($packageName, $productId, $purchaseToken);

            // purchaseState==0 یعنی خرید موفق
            if (isset($purchase->purchaseState) && $purchase->purchaseState === 0) {
                return [
                    'success'  => true,
                    'purchase' => $purchase
                ];
            } else {
                return [
                    'success'  => false,
                    'error'    => 'purchaseState is not SUCCESS',
                    'purchase' => $purchase
                ];
            }
        } catch (\Google\Service\Exception $e) {
            return [
                'success' => false,
                'error'   => $e->getMessage(),
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error'   => $e->getMessage(),
            ];
        }
    }

    /**
     * تأیید خرید (در صورت نیاز)
     */
    public function verify(Request $request, Purchase $purchase): JsonResponse
    {
        $request->validate([
            'status'      => 'required|in:success,failed',
            'fail_reason' => 'nullable|string|max:255',
        ]);

        if ($request->status === 'success') {
            $this->verifyPurchaseOwnership($purchase);
        }

        $purchase->update([
            'status'      => $request->status,
            'fail_reason' => $request->fail_reason,
        ]);

        return response()->json([
            'message'  => 'Purchase verified',
            'purchase' => $purchase,
        ]);
    }

    /**
     * تایید مالکیت خرید
     */
    private function verifyPurchaseOwnership(Purchase $purchase): void
    {
        if (!$purchase->device_id) {
            return;
        }

        $user = User::where('username', $purchase->device_id)->first();

        if (!$user) {
            $this->createFraudAlert(
                $purchase,
                'FRAUD_DEVICE_NOT_FOUND',
                "Device ID '{$purchase->device_id}' پیدا نشد در سیستم"
            );
            $purchase->update([
                'status'      => 'fraud',
                'fail_reason' => 'Device not found in system'
            ]);
            return;
        }

        $user->increment('credit_seconds', $purchase->seconds_added);
        $user->update(['credit_updated_at' => now()]);

        $purchase->update([
            'user_id' => $user->id,
            'status'  => 'success'
        ]);
    }

    /**
     * ثبت هشدار تقلب
     */
    private function createFraudAlert(Purchase $purchase, string $type, string $description): void
    {
        FraudAlert::create([
            'purchase_id' => $purchase->id,
            'device_id'   => $purchase->device_id,
            'user_id'     => $purchase->user_id,
            'alert_type'  => $type,
            'description' => $description,
            'ip_address'  => $purchase->ip,
            'user_agent'  => $purchase->user_agent,
            'status'      => 'pending',
            'created_at'  => now(),
        ]);
    }

    /**
     * لیست خریدها
     */
    public function index(): JsonResponse
    {
        $purchases = Purchase::with('user')->latest()->paginate(20);
        return response()->json($purchases);
    }

    /**
     * نمایش یک خرید خاص
     */
    public function show(Purchase $purchase): JsonResponse
    {
        return response()->json($purchase->load('user'));
    }

    /**
     * لیست خریدهای یک کاربر
     */
    public function userPurchases(User $user): JsonResponse
    {
        $purchases = $user->purchases()->latest()->paginate(20);
        return response()->json($purchases);
    }
}