<?php
// File: /home/okgroups/public_html/sm.okgroups.com/app/Http/Controllers/Web/AppContentsController.php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AppContentsController extends Controller
{
    // 21 languages from app
    private array $langs = [
        ['name' => 'فارسی', 'code' => 'fa'],
        ['name' => 'English', 'code' => 'en'],
        ['name' => 'العربية', 'code' => 'ar'],
        ['name' => 'Français', 'code' => 'fr'],
        ['name' => 'Deutsch', 'code' => 'de'],
        ['name' => 'Русский', 'code' => 'ru'],
        ['name' => 'Italiano', 'code' => 'it'],
        ['name' => 'Español', 'code' => 'es'],
        ['name' => 'Türkçe', 'code' => 'tr'],
        ['name' => '日本語', 'code' => 'ja'],
        ['name' => '中文', 'code' => 'zh'],
        ['name' => 'हिन्दी', 'code' => 'hi'],
        ['name' => '한국어', 'code' => 'ko'],
        ['name' => 'اردو', 'code' => 'ur'],
        ['name' => 'Português', 'code' => 'pt'],
        ['name' => 'Nederlands', 'code' => 'nl'],
        ['name' => 'Svenska', 'code' => 'sv'],
        ['name' => 'Ελληνικά', 'code' => 'el'],
        ['name' => 'Română', 'code' => 'ro'],
        ['name' => 'עברית', 'code' => 'he'],
        ['name' => 'Kurdî', 'code' => 'ku'],
    ];

    public function index(Request $request)
    {
        $selectedKey = $request->query('key');

        // Keys list (show the ones that exist)
        $keys = DB::table('app_contents')
            ->select('key')
            ->distinct()
            ->orderBy('key')
            ->pluck('key')
            ->values()
            ->all();

        if (!$selectedKey) {
            // Prefer these common keys if exist
            $preferred = ['privacy', 'terms', 'about', 'version_popup', 'voucher_popup'];
            foreach ($preferred as $p) {
                if (in_array($p, $keys, true)) { $selectedKey = $p; break; }
            }
            if (!$selectedKey && count($keys) > 0) $selectedKey = $keys[0];
        }

        // Fetch all rows for selected key
        $rows = [];
        if ($selectedKey) {
            $rows = DB::table('app_contents')
                ->where('key', $selectedKey)
                ->get()
                ->keyBy('language_code')
                ->toArray();
        }

        // Prepare per-language form values
        $items = [];
        foreach ($this->langs as $l) {
            $lc = $l['code'];
            $row = $rows[$lc] ?? null;

            $items[$lc] = [
                'title' => $row?->title ?? '',
                'body'  => $row?->body ?? '',
            ];
        }

        // Base meta defaults (inherit from en if exists; else first available)
        $base = $rows['en'] ?? (count($rows) ? reset($rows) : null);
        $meta = [
            'format'      => $base?->format ?? 'html',
            'is_active'   => (int)($base?->is_active ?? 1),
            'is_verified' => (int)($base?->is_verified ?? 1),
        ];

        return view('settings.app_contents', [
            'keys'        => $keys,
            'selectedKey' => $selectedKey,
            'langs'       => $this->langs,
            'items'       => $items,
            'meta'        => $meta,
        ]);
    }

    public function update(Request $request)
    {
        $key = trim((string)$request->input('key', ''));

        if ($key === '') {
            return back()->withErrors(['key' => 'Key is required.'])->withInput();
        }

        $items = $request->input('items', []);
        if (!is_array($items)) $items = [];

        // meta (keep stable across languages)
        $format      = $request->input('meta.format', 'html') ?: 'html';
        $isActive    = (int)($request->input('meta.is_active', 1));
        $isVerified  = (int)($request->input('meta.is_verified', 1));

        // If key exists, inherit meta from existing EN row (safest)
        $existingEn = DB::table('app_contents')
            ->where('key', $key)
            ->where('language_code', 'en')
            ->first();

        if ($existingEn) {
            $format     = $existingEn->format ?? $format;
            $isActive   = (int)($existingEn->is_active ?? $isActive);
            $isVerified = (int)($existingEn->is_verified ?? $isVerified);
        }

        // Voucher rule: body MUST keep {code}
        if ($key === 'voucher_popup') {
            foreach ($this->langs as $l) {
                $lc = $l['code'];
                $body = (string)($items[$lc]['body'] ?? '');
                if ($body !== '' && strpos($body, '{code}') === false) {
                    return back()->withErrors([
                        "items.$lc.body" => "For voucher_popup, the body must contain {code} (do not remove it). Language: $lc"
                    ])->withInput();
                }
            }
        }

        DB::beginTransaction();
        try {
            foreach ($this->langs as $l) {
                $lc = $l['code'];

                $title = (string)($items[$lc]['title'] ?? '');
                $body  = (string)($items[$lc]['body'] ?? '');

                // If completely empty AND row doesn't exist, skip creating noise
                $exists = DB::table('app_contents')
                    ->where('key', $key)
                    ->where('language_code', $lc)
                    ->exists();

                if (!$exists && trim($title) === '' && trim($body) === '') {
                    continue;
                }

                DB::table('app_contents')->updateOrInsert(
                    ['key' => $key, 'language_code' => $lc],
                    [
                        'title'       => $title,
                        'body'        => $body,
                        'format'      => $format,
                        'is_active'   => $isActive,
                        'is_verified' => $isVerified,
                        'updated_at'  => now(),
                        'created_at'  => $exists ? DB::raw('created_at') : now(),
                    ]
                );
            }

            DB::commit();
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->withErrors(['save' => 'Save failed: ' . $e->getMessage()])->withInput();
        }

        return redirect()
            ->route('settings.app_contents.index', ['key' => $key])
            ->withSuccess('App contents saved successfully.');
    }
}