<?php
// File: /home/okgroups/public_html/sm.okgroups.com/app/Http/Controllers/Web/VouchersController.php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Voucher;
use App\Models\VoucherRedemption;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class VouchersController extends Controller
{
    public function index(Request $request)
    {
        $q = Voucher::query();

        if ($request->filled('status')) {
            $status = $request->string('status')->toString();

            if ($status === 'active') {
                $q->where('is_active', true);
            } elseif ($status === 'inactive') {
                $q->where('is_active', false);
            } elseif ($status === 'used') {
                $q->whereColumn('uses_count', '>=', 'max_uses');
            } elseif ($status === 'expired') {
                $q->whereNotNull('expires_at')->where('expires_at', '<', now());
            }
        }

        if ($request->filled('search')) {
            $search = trim($request->string('search')->toString());
            $q->where('code', 'like', '%' . $search . '%');
        }

        $vouchers = $q->orderByDesc('id')->paginate(25)->withQueryString();

        return view('settings.vouchers.index', compact('vouchers'));
    }

    public function create()
    {
        return view('settings.vouchers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => ['nullable', 'string', 'max:64'],
            'seconds' => ['required', 'integer', 'min:1'],
            'price' => ['nullable', 'numeric', 'min:0'],
            'title' => ['nullable', 'string', 'max:120'],
            'max_uses' => ['nullable', 'integer', 'min:1'],
            'expires_at' => ['nullable', 'date'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $code = $validated['code'] ?? null;
        $code = $code ? strtoupper(trim($code)) : strtoupper(Str::random(10));

        // ensure unique (in case random collides)
        while (Voucher::where('code', $code)->exists()) {
            $code = strtoupper(Str::random(10));
        }

        $voucher = Voucher::create([
            'code' => $code,
            'seconds' => (int)$validated['seconds'],
            'price' => $validated['price'] ?? null,
            'title' => $validated['title'] ?? null,
            'max_uses' => isset($validated['max_uses']) ? (int)$validated['max_uses'] : 1,
            'uses_count' => 0,
            'is_active' => $request->boolean('is_active', true),
            'expires_at' => $validated['expires_at'] ?? null,
            'created_by' => auth()->id(),
        ]);

        return redirect()
            ->route('settings.vouchers.show', $voucher->id)
            ->with('success', 'Voucher created successfully.');
    }

    public function show(Voucher $voucher)
    {
        $redemptions = VoucherRedemption::query()
            ->where('voucher_id', $voucher->id)
            ->orderByDesc('id')
            ->paginate(25);

        return view('settings.vouchers.show', compact('voucher', 'redemptions'));
    }
}
